<?php
/**
 * Google Sheets Integration
 * Syncs attendance data to Google Sheets
 * 
 * SETUP INSTRUCTIONS:
 * 1. Go to Google Cloud Console (https://console.cloud.google.com/)
 * 2. Create a new project or select existing one
 * 3. Enable Google Sheets API
 * 4. Create Service Account credentials
 * 5. Download JSON key file
 * 6. Share your Google Sheet with the service account email
 * 7. Update the configuration below
 */

// Google Sheets Configuration
define('GOOGLE_SHEETS_ENABLED', false); // Set to true to enable
define('GOOGLE_SHEETS_ID', ''); // Your Google Sheet ID (from URL)
define('GOOGLE_SHEETS_RANGE', 'Sheet1!A:E'); // Range to write data
define('GOOGLE_SERVICE_ACCOUNT_FILE', __DIR__ . '/google_service_account.json'); // Path to JSON key file

/**
 * Append attendance to Google Sheets
 * @param string $employeeName
 * @param string $department
 * @param string $date
 * @param string $checkInTime
 * @param string $checkOutTime
 * @return bool
 */
function appendToGoogleSheets($employeeName, $department, $date, $checkInTime, $checkOutTime = '') {
    if (!GOOGLE_SHEETS_ENABLED || empty(GOOGLE_SHEETS_ID)) {
        return false;
    }
    
    // Check if service account file exists
    if (!file_exists(GOOGLE_SERVICE_ACCOUNT_FILE)) {
        error_log("Google Sheets: Service account file not found");
        return false;
    }
    
    try {
        // Load service account credentials
        $credentials = json_decode(file_get_contents(GOOGLE_SERVICE_ACCOUNT_FILE), true);
        
        if (!$credentials) {
            error_log("Google Sheets: Invalid service account file");
            return false;
        }
        
        // Get access token
        $accessToken = getGoogleAccessToken($credentials);
        
        if (!$accessToken) {
            error_log("Google Sheets: Failed to get access token");
            return false;
        }
        
        // Prepare data
        $values = [[
            $date,
            $employeeName,
            $department,
            $checkInTime,
            $checkOutTime
        ]];
        
        // Append to sheet
        $url = "https://sheets.googleapis.com/v4/spreadsheets/" . GOOGLE_SHEETS_ID . "/values/" . GOOGLE_SHEETS_RANGE . ":append?valueInputOption=RAW";
        
        $ch = curl_init($url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Authorization: Bearer ' . $accessToken,
            'Content-Type: application/json'
        ]);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode([
            'values' => $values
        ]));
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($httpCode === 200) {
            return true;
        } else {
            error_log("Google Sheets API error: " . $response);
            return false;
        }
        
    } catch (Exception $e) {
        error_log("Google Sheets error: " . $e->getMessage());
        return false;
    }
}

/**
 * Get Google Access Token using Service Account
 * @param array $credentials
 * @return string|false
 */
function getGoogleAccessToken($credentials) {
    $jwt = createJWT($credentials);
    
    $ch = curl_init('https://oauth2.googleapis.com/token');
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query([
        'grant_type' => 'urn:ietf:params:oauth:grant-type:jwt-bearer',
        'assertion' => $jwt
    ]));
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($httpCode === 200) {
        $data = json_decode($response, true);
        return isset($data['access_token']) ? $data['access_token'] : false;
    }
    
    return false;
}

/**
 * Create JWT for Google Service Account
 * @param array $credentials
 * @return string
 */
function createJWT($credentials) {
    $header = [
        'alg' => 'RS256',
        'typ' => 'JWT'
    ];
    
    $now = time();
    $payload = [
        'iss' => $credentials['client_email'],
        'scope' => 'https://www.googleapis.com/auth/spreadsheets',
        'aud' => 'https://oauth2.googleapis.com/token',
        'exp' => $now + 3600,
        'iat' => $now
    ];
    
    $headerEncoded = base64UrlEncode(json_encode($header));
    $payloadEncoded = base64UrlEncode(json_encode($payload));
    
    $signature = '';
    $data = $headerEncoded . '.' . $payloadEncoded;
    
    if (openssl_sign($data, $signature, $credentials['private_key'], OPENSSL_ALGO_SHA256)) {
        $signatureEncoded = base64UrlEncode($signature);
        return $data . '.' . $signatureEncoded;
    }
    
    return '';
}

/**
 * Base64 URL encode
 * @param string $data
 * @return string
 */
function base64UrlEncode($data) {
    return rtrim(strtr(base64_encode($data), '+/', '-_'), '=');
}
